/* ==================================================================
File        : lensFilter.hlsl
Author      : Hlynur Tryggvason
Date        : 2009
Format      : HLSL/Cg
Description : Renders a texture to a screen space quad which is warped in a way to simulate a camera lens

================================================================== */

#include "default.hlsl"

//------------------------------------------------------------------------------------------------
// Shader constants
//------------------------------------------------------------------------------------------------
uniform float3 g_lensFilterParams	: register(c0); // scaleU, scaleV, distortion

//------------------------------------------------------------------------------------------------
// Distortion function
//------------------------------------------------------------------------------------------------

// Input:  Vertex position in the range [-1,+1]
// Output: Distorted position in the range [-1,+1]
float2 CalculateDistortion( float2 pos, float k )
{
	// Undistorted radius.
	float u = sqrt( pos.x * pos.x + pos.y * pos.y );

#if defined(CG_PS3)	
	// Points around origin become a bit destorted on PS3
	if( u < 0.1f )
		return pos;
#endif

	//if( u < 0.5f )
	//	return pos;

	// Distorted radius.
	float t1 = sqrt( (27.0f * k * u * u + 4.0f) / k );
	const float sqrt_3 = 1.7320508076f;
	float t2 = 6.0f * sqrt_3 * k;
	float t3 = u / ( 2.0f * k );
	float t4 = pow( abs((t1/t2) + t3), 0.3333f );
	float d  = t4 - 1.0f / ( 3.0f * k * t4 );

	// Calculate scale
#if defined(_NEGATIVE_DISTORTION)
	float scale = d / u;
#else
	float scale = u / d;
#endif

	// Return the distorted value
	return pos * scale * g_lensFilterParams.xy;
}

//------------------------------------------------------------------------------------------------
// Vertex shader
//------------------------------------------------------------------------------------------------

VS_OUT_POS_UVSET01 lensFilter_vx( VS_IN_2D input )
{
	VS_OUT_POS_UVSET01 Out = (VS_OUT_POS_UVSET01)0;

	Out.position = input.position;
	
#if defined(_WINPC)
	// Need to use the UVs as vertex positions
	// because the vertices have been offset
	
	// Transform from [0,1] to [-1,1]
	float2 pos = input.uvSet0.xy * 2.0f - 1.0f;			
	Out.uvSet01.xy = CalculateDistortion( pos, g_lensFilterParams.z );		
#else
	Out.uvSet01.xy = CalculateDistortion( input.position.xy, g_lensFilterParams.z );
#endif

	// Transform from [-1,1] to [0,1]
	Out.uvSet01.x = ( Out.uvSet01.x + 1.0f ) * 0.5f;

#if defined(_WINPC)
	// Don't need to flip it because we used UVs 
	Out.uvSet01.y = ( Out.uvSet01.y + 1.0f ) * 0.5f;
#else
	Out.uvSet01.y = 1.0f - ( Out.uvSet01.y + 1.0f ) * 0.5f;
#endif

	return Out;
}
